// $Id: HNSCocoaScript.m,v 1.15 2003/03/20 13:59:39 hns Exp $
//
//
//  CocoaScript.m
//  CocoaScript
//
//  Created by Dr. H. Nikolaus Schaller on Sat Jun 15 2002.
//  Copyright (c) 2001 __MyCompanyName__. All rights reserved.
//

#import <HNSCocoaScript.h>
#import <HNSAnsel.h>

@interface HNSCocoaScript (PrivateExtensions)
+ (HNSCocoaScript *) scriptForSelector:(SEL) s;	// make method call
+ (NSString *) scanName;
+ (NSString *) scanOperator;
+ (id) scanObject;			// basic object
+ (id) scanPobject;			// parenthetized object
+ (id) scanSequence;		// operator/method sequence
+ (id) scanMethodCall;		// method call
- (void) setLeft:(id) l;
- (void) setMethod:(SEL) sel;
- (void) addRight:(id) r;
@end

@implementation HNSCocoaScript

// used for debugging/single stepper mode
static NSMutableArray *sharedStack;			// common stack
static id accu;								// current value
static HNSCocoaScript *currentScript;		// current script
static NSArray *currentCode;				// current statements
static unsigned int pc;						// current PC (instruction pointer)
static unsigned int epc;					// where to end PC
static NSArray *currentArgv;				// current script arguments
static NSMutableDictionary *gvars;			// variables

// used during scanFromScanner
static NSScanner *scanner;

//
// CocoaScript Runtime Interpreter
//

+ (id) keyword:(NSString *) sym
{ // get keyword ressource
	static NSDictionary *kw;
	if(kw == nil)
		{ // load from resources file
		NSString *sympath=[[NSBundle bundleForClass:[self class]] pathForResource:@"HNSCocoaScript" ofType:@"plist"];
		kw=[[NSDictionary dictionaryWithContentsOfFile:sympath] retain];
#if 0
		NSLog(@"dictionary %@ loaded from %@", sym, sympath);
#endif
		}
	return [kw objectForKey:sym];
}

+ (void) defineMethod:(NSString *) method ofType:(Class) type andArgs:(NSArray *) arg asScript:(HNSCocoaScript *) code;	// make script called when (class/instance) method is invoked
{ // make script called when (class/instance) method is invoked
	// create dictionary for classes and methods
}

- (id) eval
{
	return [self evaluate:[NSArray arrayWithObjects:self, NSStringFromSelector(_cmd), nil]];	// no args
}

#if ENCODING
Table 4-1 Objective-C type encodings
Code Meaning
c A char
i An int
s A short
l A long
q A long long
C An unsigned char
I An unsigned int
S An unsigned short
L An unsigned long
Q An unsigned long long
f A float
d A double
v A void
* A character string (char *)
@ An object (whether statically typed or typed id)
// # A class object (Class)
: A method selector (SEL)
[array type] An array
{name=type...} A structure
(type...) A union
bnum A bit field of num bits
^type A pointer to type
? An unknown type (among other things, this code is used for function pointers)

The type code for an array is enclosed within square brackets; the number of elements in the array is specified immediately after the open bracket, before the array type. For example, an array of 12 pointers to floats would be encoded as:
[12^f]
Structures are specified within braces, and unions within parentheses. The structure tag is listed first, followed by an equal sign and the codes for the fields of the structure listed in sequence. For example, this structure,
typedef struct example {
	id   anObject;
	char *aString;
	int  anInt;
} Example;
would be encoded like this:
{example=@*i}
The same encoding results whether the defined type name (Example) or the structure tag (example) is passed to @encode(). The encoding for a structure pointer carries the same amount of information
about the structures fields:
^{example=@*i}
However, another level of indirection removes the internal type specification:
^^{example}
Objects are treated like structures. For example, passing the NSObject class name to @encode() yields this encoding:
{NSObject=#}
The NSObject class declares just one instance variable, isa, of type Class.
Note that although the @encode() directive doesnt return them, the runtime system uses these additional encodings for type qualifiers when theyre used to declare methods in a protocol:

Table 4-2 Objective-C method encodings
Code Meaning
r const
n in
N inout
o out
O bycopy
R byref
V oneway

// from NSInvocation.h

NSObjCNoType = 0,
NSObjCVoidType = 'v',
NSObjCCharType = 'c',
NSObjCShortType = 's',
NSObjCLongType = 'l',
NSObjCLonglongType = 'q',
NSObjCFloatType = 'f',
NSObjCDoubleType = 'd',
#if MAC_OS_X_VERSION_10_2 <= MAC_OS_X_VERSION_MAX_ALLOWED
NSObjCBoolType = 'B',
#endif
NSObjCSelectorType = ':',
NSObjCObjectType = '@',
NSObjCStructType = '{',
NSObjCPointerType = '^',
NSObjCStringType = '*',
NSObjCArrayType = '[',
NSObjCUnionType = '(',
NSObjCBitfield = 'b'

#endif

- (void) copyEvalArgument:(const char **) type buffer:(void **) bfr value:(id) accu
{ // convert argument to expected type (may be recursive for structs)
#if 0
	NSLog(@"copyEvalArgument:(%s) buffer:%08lX value:%@", *type, *bfr, accu);
#endif
	switch(*(*type)++)
		{ // decode first character of type
		case NSObjCObjectType:	// plain object
			{
				*((id *) *bfr)++=accu;
				break;
			}
		case NSObjCCharType:	// signed char
			{
				*((char *) *bfr)++=[accu intValue];
				break;
			}
		case NSObjCShortType:	// signed short
			{
				*((short *) *bfr)++=[accu intValue];
				break;
			}
		case 'i':	// signed integer
			{
				*((int *) *bfr)++=[accu intValue];
				break;
			}
		case 'I':	// unsigned integer
			{
				*((unsigned *) *bfr)++=[accu intValue];
				break;
			}
		case NSObjCLongType:	// signed long
			{
				*((long *) *bfr)++=[accu intValue];
				break;
			}
		case NSObjCFloatType:	// float
			{
#if 0
				NSLog(@"bfr=%08lx", bfr);
				NSLog(@"*bfr=%08lx", *bfr);
				NSLog(@"*(float *) bfr)=%08lx", *bfr);
#endif
				*((float *) *bfr)++=[accu floatValue];
				break;
			}
		case NSObjCDoubleType:	// double
			{
				*((double *) *bfr)++=[accu doubleValue];
				break;
			}
		case NSObjCSelectorType:
			{ // handle selectors - e.g. for setAction:
				*((SEL *) *bfr)++=NSSelectorFromString(accu);	// should better be a string!
				break;
			}
		case '#':
			{ // handle class
				*((id *) *bfr)++=[accu class];	// should better be a class object that responds to +class!
				break;
			}
		case NSObjCStructType:
			{ // handle structures (nested!)
				int i;
				if(![accu respondsToSelector:@selector(objectAtIndex:)])
					[NSException raise:@"Value Error" format:@"method expects structure/array argument"];
				while(**type != '=')
					{
					if(**type == 0)
						[NSException raise:@"Internal Error" format:@"invalid type string"];
					(*type)++;
					}
				(*type)++;	// skip =
				i=0;
				while(**type != '}')
					{
					if(**type == 0)
						[NSException raise:@"Internal Error" format:@"invalid type string"];
#if 0
					NSLog(@"before %d: %08lX", i, *bfr);
#endif
					[self copyEvalArgument:type buffer:bfr value:[accu objectAtIndex:i++]];	// extract array
#if 0
					NSLog(@"after %d: %08lX", i, *bfr);
#endif
					}
				(*type)++;	// skip }
				break;
			}
		case NSObjCPointerType:
			{ // handle pointers
			}
		case NSObjCUnionType:	// union not supported
		default:
			[NSException raise:@"Internal Error" format:@"can't process argument type (%s)", type];
		}
}

- (id) evaluate:(NSArray *) args
{
	id accu;
	int i;
#if 0
	NSLog(@"eval args: %@", args);
	NSLog(@"left:      %@", left);
    NSLog(@"method:    %@", NSStringFromSelector(method));
	NSLog(@"right:     %@", right);
#endif
#if 0
	if(debugging)
		{ // enter debugging mode
		}
#endif
	if(!left)
		{ // no left - decode some special codes
		if(method == @selector(getArgv))
			return args;	// special
		if(method == @selector(getScript))
			return self;	// special
		if(method == @selector(setValueForName:to:))
			{ // special
			id key=[right objectAtIndex:0];	// get Name argument
			if([key respondsToSelector:@selector(eval)])
				key=[key eval];
			accu=[right objectAtIndex:1];	// get To argument
			if([accu respondsToSelector:@selector(eval)])
				accu=[accu eval];
			[self setValueForName:key to:accu];
			return accu;
			}
		if(method == @selector(valueForName:))
			{
			id key=[right objectAtIndex:0];	// get Name argument
			if([key respondsToSelector:@selector(eval)])
				key=[key eval];
			return [self valueForName:key];
			}
		if(method == @selector(makeArray:))
			{ // build fresh array from contents
			NSMutableArray *a=[NSMutableArray arrayWithCapacity:[right count]];
			for(i=0; i<[right count]; i++)
				{ // evaluate all arguments
				accu=[right objectAtIndex:i];
				if([accu respondsToSelector:@selector(eval)])
					accu=[accu eval];
				[a addObject:accu];
				}
			return a;
			}
		if(method == @selector(makeDictionary:))
			{ // build fresh dictionary from contents
			NSMutableDictionary *a=[NSMutableDictionary dictionaryWithCapacity:[right count]/2];
			id o;
			for(i=0; i<[right count]; i+=2)
				{ // evaluate all arguments
				accu=[right objectAtIndex:i];
				if([accu respondsToSelector:@selector(eval)])
					accu=[accu eval];
				o=[right objectAtIndex:i+1];
				if([o respondsToSelector:@selector(eval)])
					o=[o eval];
				[a setObject:o forKey:accu];
				}
			return a;
			}
		}
	if(method == @selector(CSEVALSHIELD))
		return left;	// done
	if(left && [left respondsToSelector:@selector(eval)])
 		accu=[left eval];
 	else
		accu=left;	// default: keep left unevaluated - may be nil
	if(method)
		{ // method call
		const char *type;
		void *bfr, *bp;
		NSMethodSignature *ms;
		NSInvocation *ir;	// create invocation record
#if 0
		NSLog(@"[%@ %@...]", accu, NSStringFromSelector(method));
#endif
		if(![accu respondsToSelector:method])
			[NSException raise:@"Internal Error" format:@"object '%@' does not respond to selector '%@'", accu, NSStringFromSelector(method)];
		ms=[accu methodSignatureForSelector:method];
		if(ms == nil)
			[NSException raise:@"Internal Error" format:@"method '%@' has no signature for object '%@'", NSStringFromSelector(method), accu];
		bfr=malloc([ms frameLength]);	// should be enough for a single argument
		if([right count]+2 < [ms numberOfArguments])
			[NSException raise:@"Internal Error" format:@"not enough arguments (%d) specified for method '%@'", [right count], NSStringFromSelector(method)];
		ir=[NSInvocation invocationWithMethodSignature:ms];
		[ir setTarget:accu];
		[ir setSelector:method];
		for(i=0; i<[right count]; i++)
			{ // evaluate all arguments
			accu=[right objectAtIndex:i];
			if([accu respondsToSelector:@selector(eval)])
				accu=[accu eval];
			if(i+2 >= [ms numberOfArguments])
				{ // is vararg
				[ir setVariableArgument:accu ofType:NSObjCObjectType];
				continue;
				}
			type=[ms getArgumentTypeAtIndex:(i+2)];
#if 0
			NSLog(@"arg type: %s", type);
#endif
#if CHECK_STRUCT
			if(*type == '{')
				{ // check struct processing
				[self copyEvalArgument:&type buffer:&bp value:accu];	// copy (recursively) to buffer
				NSLog(@"NSRect=(%f, %f, %f, %f)", ((float *) bfr)[0], ((float *) bfr)[1], ((float *) bfr)[2], ((float *) bfr)[3]);
				}
				else
#endif
			bp=bfr;
			[self copyEvalArgument:&type buffer:&bp value:accu];	// copy (recursively) to buffer
			[ir setArgument:bfr atIndex:(i+2)];
#if OLD
			switch(*type)
				{ // convert object
				case NSObjCObjectType:	// plain object
					[ir setArgument:&accu atIndex:(i+2)];
					break;
				case 'c':	// signed char
					{
						char v=[accu intValue];
						[ir setArgument:&v atIndex:(i+2)];
						break;
					}
				case 's':	// signed short
					{
						short v=[accu intValue];
						[ir setArgument:&v atIndex:(i+2)];
						break;
					}
				case 'i':	// signed integer
					{
						int v=[accu intValue];
						[ir setArgument:&v atIndex:(i+2)];
						break;
					}
				case 'I':	// unsigned integer
					{
						unsigned v=[accu intValue];
						[ir setArgument:&v atIndex:(i+2)];
						break;
					}
				case 'l':	// signed long
					{
						long v=[accu intValue];
						[ir setArgument:&v atIndex:(i+2)];
						break;
					}
				case 'f':	// float
					{
						float v=[accu floatValue];
						[ir setArgument:&v atIndex:(i+2)];
						break;
					}
				case 'd':	// double
					{
						double v=[accu doubleValue];
						[ir setArgument:&v atIndex:(i+2)];
						break;
					}
				case ':':
					{ // handle selectors - e.g. for setAction:
						SEL v=NSSelectorFromString(accu);	// should better be a string!
						[ir setArgument:&v atIndex:(i+2)];
						break;
					}
				case '#':
					{ // handle class
						Class v=[accu class];	// should better be a class object that responds to +class!
						[ir setArgument:&v atIndex:(i+2)];
						break;
					}
				case '{':
					{ // handle structures (nested!)
					}
				case '^':
					{ // handle pointers
					}
				case '(':	// union not supported
				default:
					[NSException raise:@"Internal Error" format:@"can't process argument type (%s) of argument %d of method '%@'", type, i, NSStringFromSelector(method)];
				}
#endif
			}
		[ir invoke];
		type=[ms methodReturnType];
        free(bfr);
#if 0
		NSLog(@"return type: %s", type);
#endif
		switch(*type)
			{ // get return value
			case '#':	// Class
			case NSObjCObjectType:	// object
				[ir getReturnValue:&accu];
				return accu;
			case NSObjCVoidType:	// void
				return nil;
			case NSObjCCharType:	// character sized integer
				{
					char i;
					[ir getReturnValue:&i];
					return [NSNumber numberWithInt:i];
				}
			case NSObjCShortType:	// short sized integer
				{
					short i;
					[ir getReturnValue:&i];
					return [NSNumber numberWithInt:i];
				}
			case 'i':	// integer
				{
					int i;
					[ir getReturnValue:&i];
					return [NSNumber numberWithInt:i];
				}
			case 'I':	// unsigned integer
				{
					unsigned int i;
					[ir getReturnValue:&i];
					return [NSNumber numberWithInt:i];
				}
			case NSObjCLongType:	// integer
				{
					long i;
					[ir getReturnValue:&i];
					return [NSNumber numberWithInt:i];
				}
			case NSObjCSelectorType:	// selector
				{
					SEL i;
					[ir getReturnValue:&i];
					return NSStringFromSelector(i);
				}
			case NSObjCStructType:	// struct { _name=type list } - should make NSArray out of it
				{
				}
			case NSObjCUnionType:	// union not supported
			default:
				[NSException raise:@"Internal Error" format:@"can't process return type (%s) of method '%@'", type, NSStringFromSelector(method)];
			}
		}
	// plain script execution
	for(i=0; i<[right count]; i++)
		{ // evaluate all statements; return value of last one
		accu=[right objectAtIndex:i];	// get statement
#if 0
		NSLog(@"execute statement: %@", accu);
#endif
		if(accu && [accu respondsToSelector:@selector(eval)])
			accu=[accu eval];
		}
	return accu;
}

- (void) CSloop;
{ // loop current script until breaked
	NS_DURING	// set up exception handler!
		while(YES)
			{
#if 0
			NSLog(@"CSloop: %@", self);
#endif
			[self eval];
			}
	NS_HANDLER
#if 0
		NSLog(@"CSloop: catched exception: %@ - %@", [localException name], [localException reason]);
#endif
		if(![[localException name] isEqualToString:@"HNSBreakLoop"])
			[localException raise]; 	// Re-raise the exception
	NS_ENDHANDLER
}

- (void) CSexception:(HNSCocoaScript *) handler;
{ // catch exceptions in block and call handler
	NS_DURING	// set up exception handler!
		[self eval];
	NS_HANDLER
#if 0
		NSLog(@"CSexception: catched exception: %@ - %@", [localException name], [localException reason]);
#endif
		if([[localException name] isEqualToString:@"HNSBreakLoop"] || [[localException name] isEqualToString:@"HNSReturn"])
			[localException raise]; 	// Re-raise the exception
		[handler evaluate:[NSArray arrayWithObjects:handler, NSStringFromSelector(_cmd), localException, nil]];	// pass exception record
// needs mechanism to handle		[localException raise]; 	// Re-raise the exception
	NS_ENDHANDLER
}

- (id) CSfunction;
{ // catch exceptions in block and call handler
	id val;
	NSMutableDictionary *vsave=gvars;	// save local variables
#if 0
	NSLog(@"eval procedure: %@", self);
#endif
	gvars=[[NSMutableDictionary dictionaryWithCapacity:10] retain];	// create new variables
	NS_DURING	// set up exception handler!
		[self eval];	// evaluate
	NS_HANDLER
#if 0
		NSLog(@"CSfunction: catched exception: %@ - %@", [localException name], [localException reason]);
#endif
		if(![[localException name] isEqualToString:@"HNSReturn"])
		    [localException raise]; 	// Re-raise the exception
	NS_ENDHANDLER
	val=[gvars objectForKey:@"retval"];	// fetch return value (may be nil!)
	[gvars release];
	gvars=vsave;		// restore variables
	return val;
}

- (id) CSEVALSHIELD;								// returns script object instead of result
{ // should not evaluate!
	[NSException raise:@"Internal Error" format:@"script shielded from evaluation"];
	return self;
}

//
// CocoaScript Runtime Debugger
//

+ (NSArray *) stack;
{ // return shared program flow stack
	if(!sharedStack)
		sharedStack=[[NSMutableArray arrayWithCapacity:100] retain];
	return sharedStack;
}

+ (void) clearStack;
{ // clear shared stack
	[self stack];	// create a new stack if required
	[sharedStack removeAllObjects];
}

- (NSString *) description;
{
	if(left)
		{
		if(method)
			{
			if([right count])
				return [NSString stringWithFormat:@"[%@ %@%@]", left, NSStringFromSelector(method), [right description]];
			else
				return [NSString stringWithFormat:@"[%@ %@]", left, NSStringFromSelector(method)];
			}
		return [NSString stringWithFormat:@"%@", left];
		}
	if(method)
		return [NSString stringWithFormat:@"[nil %@%@]", NSStringFromSelector(method), [right description]];
	return [NSString stringWithFormat:@"%@", [right description]];
}

- (void) invoke:(NSArray *) args
{ // invoke this script
	// push
	accu=nil;
	currentScript=self;
	if(!right)
		[NSException raise:@"Internal Error" format:@"init message missing"];
	currentCode=right;
	pc=0;
	epc=[right count];
	currentArgv=args;
}

- (BOOL) singleStep;
{ // evaluate one single step
	id cmd;
	if(pc >= epc)
		return NO;		// end of block reached
	cmd=[currentCode objectAtIndex:pc++];	// get next
#if 0
	NSLog(@"eval: %@", cmd);
#endif
	if([cmd isKindOfClass:[HNSCocoaScript class]])
		{ // may be method call or sub-script
		if([cmd method] != nil)
			{ // call method
			return YES;
			}
		}
	accu=cmd;	// simply push to accu
	return YES;	// may continue
}

//
// CocoaScript compiler
//

+ (NSString *) scanName
{ // symbol name
	static NSCharacterSet *sc;	// all symbol characters
	NSString *n;
	if(!sc)
		sc=[[NSCharacterSet characterSetWithCharactersInString:[HNSCocoaScript keyword:@"SymbolCharacters"]] retain];
	if([scanner scanCharactersFromSet:sc intoString:&n])
		return n;
	return nil;	// no name
}

+ (NSString *) scanOperator
{ // operator
	static NSCharacterSet *oc;	// all operator characters
	NSString *n;
	unsigned pos=[scanner scanLocation];
	if(!oc)
		oc=[[NSCharacterSet characterSetWithCharactersInString:[HNSCocoaScript keyword:@"OperatorCharacters"]] retain];
	if([scanner scanCharactersFromSet:oc intoString:&n])
		{
		if([[HNSCocoaScript keyword:@"Monops"] objectForKey:n] != nil)
			return n;
		if([[HNSCocoaScript keyword:@"Binops"] objectForKey:n] != nil)
			return n;
		[scanner setScanLocation:pos];	// back up - no valid operator
		}
	return nil;	// no valid name
}

+ (id) scanObject
{ // scan simple object
	double dbl;
	NSString *n;
	id o;
	if([scanner scanDouble:&dbl])
		{ // integer or double
		return [NSNumber numberWithDouble:dbl];
		}
	if([scanner scanString:@"\"" intoString:nil])
		{ // string constant follows
		//////// use junk scanning code from BasicCompiler
		int i;
		NSMutableString *str=[NSMutableString stringWithCapacity:50];
		for(i=[scanner scanLocation]; i<[[scanner string] length]; i++)
			{
			unichar c=[[scanner string] characterAtIndex:i];
			switch(c)
				{
				case '\\':
					i++;
					if(i == [[scanner string] length])
						break;	// end of file
						c=[[scanner string] characterAtIndex:i];	// next character
					switch(c)
						{
						case 'n':	[str appendString:@"\n"]; break;
						case 'r':	[str appendString:@"\r"]; break;
						case 'b':	[str appendString:@"\b"]; break;
						default:
							[str appendFormat:@"%C", [[scanner string] characterAtIndex:i]];	// append escaped character
						}
						continue;
				case '"':
					break;
				default:
					[str appendFormat:@"%C", c];
					continue;
				}
			break;	// end of string/file
			}
#if 0
		NSLog(@"string constant: %@", str);
#endif
		[scanner setScanLocation:i];	// set to closing " or end of string
		if(![scanner scanString:@"\"" intoString:nil])
			[NSException raise:@"Syntax Error" format:@"closing \" expected"];
		return str;
		}
	n=[self scanName];
	if(n)
		{ // yes, symbol found
		static NSDictionary *constants=nil;
		Class c;
		if([n isEqualToString:@"app"])
			return NSApp;
		if([n isEqualToString:@"self"])
			return [[HNSCocoaScript scriptForString:@"argv[0]"] autorelease];		// get indexed value
		if([n isEqualToString:@"super"])
			return [[HNSCocoaScript scriptForString:@"argv[0] class superclass"] autorelease];		// get indexed value
		if([n isEqualToString:@"nil"])
			return [HNSCocoaScript scriptForSelector:nil];	// totally empty script will return nil
		if([n isEqualToString:@"argv"])
			return [HNSCocoaScript scriptForSelector:@selector(getArgv)];
		if(!constants)
			{ // should read from symbol table resource
#define intC(x) [NSNumber numberWithInt:x], @"x"
#define boolC(x) [NSNumber numberWithBool:x], @"x"
			constants=[[NSDictionary dictionaryWithObjectsAndKeys:
			NSApp, @"app",
			[NSNumber numberWithBool:YES], @"YES",
			[NSNumber numberWithBool:NO], @"NO",
			[NSNumber numberWithDouble:1.0/0.0], @"NaN",
			//			intC(NSASCIIStringEncoding),
			// etc.
			nil] retain];
			}
		o=[constants objectForKey:n];  // NSxxx
		if(o)
			return o;	// add constant itself
		c=NSClassFromString(n);
		if(c != nil)
			return c;	// class object
		// fetch named variable
		o=[HNSCocoaScript scriptForSelector:@selector(valueForName:)];
		//	[o setLeft:self];	// this object
		[o addRight:n];		// variable name as argument
		return o;
		}
	return [self scanPobject];	// something with parentheses
}

+ (id) scanPobject;
{
	NSString *n, *op;
	id o;
	if([scanner scanString:@"(" intoString:nil])
		{ // evaluate this first
		o=[self scanSequence];
		if(![scanner scanString:@")" intoString:nil])
			[NSException raise:@"Syntax Error" format:@") expected"];
		return o;
		}
	if([scanner scanString:@"{" intoString:nil])
		{ // create embedded Object, i.e. statement list
		id p;
		o=[HNSCocoaScript scriptForSelector:nil];
		while(![scanner scanString:@"}" intoString:nil])
			{ // a list of statements
			[o addRight:[self scanSequence]];
			if([scanner scanString:@"}" intoString:nil])
				break;	// { statement } allowed
			if([scanner isAtEnd] || ![scanner scanString:@";" intoString:nil])
				[NSException raise:@"Syntax Error" format:@"; or } expected"];	// { statement; } or { statement; statement } required
			}
		p=[HNSCocoaScript scriptForSelector:@selector(CSEVALSHIELD)];	// now comes the trick - prevent evaluation
		[p setLeft:o];
		return p;
		}
	if([scanner scanString:@"[" intoString:nil])
		{ // create Array from objects
		int size=0;
		o=[HNSCocoaScript scriptForSelector:@selector(makeArray:)];
		if([scanner scanString:@"]" intoString:nil])
			{ // [] - empty array
			return o;
			}
		[o setMethod:@selector(makeArray:)];
		while(![scanner isAtEnd])
			{
			if(size == 0 && [scanner scanString:@":" intoString:nil])
				{ // [:] - empty dictionary
				[o setMethod:@selector(makeDictionary:)];
				break;
				}
			[o addRight:[self scanSequence]], size++;
			if([o method] == @selector(makeDictionary:))
				{
				if(![scanner scanString:@":" intoString:nil])
					[NSException raise:@"Syntax Error" format:@"key:value expected"];
				[o addRight:[self scanSequence]];
				}
			else if(size == 1 && [scanner scanString:@":" intoString:nil])
				{ // start dictionary mode (only permitted after first entry)
				[o setMethod:@selector(makeDictionary:)];
				[o addRight:[self scanSequence]];
				}
			if(![scanner scanString:@"," intoString:nil])
				break;
			}
		if(![scanner scanString:@"]" intoString:nil])
			[NSException raise:@"Syntax Error" format:@"] expected"];
		return o;
		}
	// must be monadic operator
	n=[self scanOperator];
	if(!n)
		[NSException raise:@"Syntax Error" format:@"Object expected"];
	op=[[HNSCocoaScript keyword:@"Monops"] objectForKey:n];
	if(!op)
		[NSException raise:@"Syntax Error" format:@"invalid unnary operator %@", n];
	o=[HNSCocoaScript scriptForSelector:NSSelectorFromString(op)];
	[o setLeft:[self scanObject]];
#if 0
	NSLog(@"monadic operator %@", n);
#endif
	return o;
}

+ (id) scanSequence;
{ // scan a statement
	NSString *n;
	id l, sc;
	l=[self scanObject];	// first part must be object
	while(YES)
		{ // operator/method call list
		NSString *op;
		if([scanner scanString:@"[" intoString:nil])
			{ // index expression follows
			HNSCocoaScript *sc;
			op=@"objectAtIndex:";
			if([scanner scanString:@":" intoString:nil])
				op=@"objectForKey:";
			sc=[HNSCocoaScript scriptForSelector:NSSelectorFromString(op)];
			[sc setLeft:l];
			[sc addRight:[self scanObject]];	// get right operand as operand into new script
			if(![scanner scanString:@"]" intoString:nil])
				[NSException raise:@"Syntax Error" format:@"missing ] in index expression"];
			l=sc;
			continue;
			}
		n=[self scanOperator];
		if(n != nil)
			{ // look up
			op=[[HNSCocoaScript keyword:@"Binops"] objectForKey:n];
#if 0
			NSLog(@"dyadic operator %@", n);
#endif
			if(op)
				{ // found
				if([op isEqualToString:@"setValueForName:to:"])
					{ // assignment - l should be valueForName:
					if([l isKindOfClass:[HNSCocoaScript class]] && [l method] == @selector(valueForName:))
						{ // ok, is a variable
						[l setMethod:@selector(setValueForName:to:)];
						[l addRight:[self scanObject]];	// get right operand as operand into new script
						continue;
						}
					[NSException raise:@"Syntax Error" format:@"left side of := is not a variable name"];
					}
				else
					{
					HNSCocoaScript *sc=[HNSCocoaScript scriptForSelector:NSSelectorFromString(op)];
					[sc setLeft:l];
					[sc addRight:[self scanObject]];	// get right operand as operand into new script
					l=sc;
					}
				continue;
				}
			[NSException raise:@"Syntax Error" format:@"invalid binary operator %@", n];
			}
		sc=[self scanMethodCall];
		if(!sc)
			break;	// no method call
		[sc setLeft:l];
		while([scanner scanString:@"," intoString:nil])
			[sc addRight:[self scanPobject]];	// additional arguments
		l=sc;
		}
	return l;
}

+ (id) scanMethodCall;
{
	NSString *n=[self scanName];
	HNSCocoaScript *o;
	if(n == nil)	// no method name follows
		return nil;
	o=[HNSCocoaScript scriptForSelector:@selector(unknown:)];
	if([scanner scanString:@":" intoString:nil])
		{ // with argument(s)
		NSString *nn;
		unsigned int pos;
		n=[n stringByAppendingString:@":"];	// add : to denote argument
		[o addRight:[self scanPobject]];	// get argument
		while(YES)
			{ // more arguments? must be name:
			pos=[scanner scanLocation];			// to back up
			nn=[self scanName];	// next part of method name?
			if(!nn)
				break;
			if(![scanner scanString:@":" intoString:nil])
				break;
			n=[n stringByAppendingFormat:@"%@:", nn];	// add next part
			[o addRight:[self scanPobject]];	// get argument
			}
		[scanner setScanLocation:pos];	// back up before last symbol not followed by :
		}
	[o setMethod:NSSelectorFromString(n)];	// full name
	return o;
}

- (void) setLeft:(id) l;
{
	[left autorelease];
	left=[l retain];
}

- (id) left;
{
	return left;
}

- (void) setMethod:(SEL) sel;
{
	method=sel;
}

- (SEL) method;
{
	return method;
}

- (void) addRight:(id) r;
{ // add to script/arguments
	if(!right)
		right=[[NSMutableArray arrayWithCapacity:10] retain];	// initialize on first argument only (saves memory)
	[right addObject:r];
}

- (NSArray *) right;
{
	return right;
}

- (void) clearScript
{ // clear script, i.e. make empty skript but keep variables
	[right removeAllObjects];
}

+ (HNSCocoaScript *) scriptFromScanner:(NSScanner *) s	// compile and return generated object script
{
	id o;
	[scanner autorelease];
	scanner=[s retain];	// keep local reference
#if 0
	NSLog(@"scanFromScanner: %@", [scanner string]);
#endif
	o=[self scanSequence];	// get first sequence
	while([scanner scanString:@";" intoString:nil])
		{ // make script of sequences
		if([o method] != nil || [o left] != nil)
			{ // o is not already a script - needs to make a script
			id p=[self scriptForSelector:nil];
			[p addRight:o];	// insert as first part
			o=p;
			}
		[o addRight:[self scanSequence]];
		}
	if(![scanner isAtEnd])
		[NSException raise:@"Syntax Error" format:@"end of script expected"];
#if 0
	NSLog(@"statements: %@", o);
#endif
	return o;
}

+ (HNSCocoaScript *) scriptForSelector:(SEL) s;
{
	HNSCocoaScript *scr;
	scr=[[[HNSCocoaScript alloc] init] autorelease];
	[scr setMethod:s];
	return scr;
}

+ (HNSCocoaScript *) scriptForString:(NSString *) s;
{
	NSScanner *sc=scanner;	// save
	id o;
	scanner=nil;			// don't release early
	o=[self scriptFromScanner:[NSScanner scannerWithString:s]];
	[scanner autorelease];
	scanner=sc;				// restore
	return o;
}

- (id) init
{
	self=[super init];
	if(self)
		{
		left=nil;
		method=nil;
		right=nil;	// [nil count] will return 0 -> does not access right
		if(!gvars)
			gvars=[[NSMutableDictionary dictionaryWithCapacity:10] retain];	// initialize
		}
	return self;
}

- (void) dealloc
{
	[left release];
	[right release];
//	[vars release];
	[super dealloc];
}

- (NSScanner *) scanner
{ // get scanner
	return scanner;
}

- (NSArray *) code
{ // get code
	return right;
}

- (void)encodeWithCoder:(NSCoder *)coder
{
	//    [super encodeWithCoder:coder];
    [coder encodeObject:left];
    [coder encodeValueOfObjCType:@encode(SEL) at:&method];
    [coder encodeObject:right];
//    [coder encodeObject:vars];
}

- (id)initWithCoder:(NSCoder *)coder
{
	//	self = [super initWithCoder:coder];
    left=[[coder decodeObject] retain];
    [coder decodeValueOfObjCType:@encode(SEL) at:&method];
    right=[[coder decodeObject] retain];
//    vars=[[coder decodeObject] retain];
	return self;
}

- (id) valueForName:(NSString *) name;					// get variable for that name
{
	// check arguments
#if 0
	NSLog(@"%@->%@\n%@", name, [gvars objectForKey:name], gvars);
#endif
	return [gvars objectForKey:name];
}

- (void) setValueForName:(NSString *) name to:(id) value;	// set variable for that name
{
#if 0
	NSLog(@"%@:=%@\n%@", name, value, gvars);
#endif
	if(value)
		[gvars setObject:value forKey:name];
	else
		[gvars removeObjectForKey:name];	// assigning nil deletes variable
}

- (NSString *) makeObjectiveC:(id) object;
{ // make for that object
#if 0
	NSString *c;
#endif
	if([object respondsToSelector:@selector(makeObjectiveC)])
		return [object makeObjectiveC];	// has its own procedure
#if 0
	c=NSStringFromClass(left);	// get name of class
	if([c count] != 0)
		s=[NSString stringWithFormat:@"[%@ %@", c, [e nextObject]];
#endif
	return @"?";
}

- (NSString *) makeObjectiveC;
{
	int i;
	NSString *s=@"";
	if(left && method)
		{ // method call
		NSString *m;
		NSEnumerator *e;
		if(method == @selector(noEval))
			{ // special processing, ignore or can't convert!!!
			return [self makeObjectiveC:left];
			}
		// check for Obj-C operators and NSNumber arguments
		if(method == @selector(CSloop))
			return [NSString stringWithFormat:@"while(YES) {\n%@}\n", [self makeObjectiveC:left]];
		if(method == @selector(CSexception:))
			return [NSString stringWithFormat:@"NS_DURING\n%@\nNS_HANDLER\n%@\nNS_ENDHANDLER\n", [self makeObjectiveC:left], [self makeObjectiveC:[right objectAtIndex:0]]];
		// check for CSprocedure method
		// check for eval/evaluate method
		// decode [NSException raise:@"HNSbreak" and raise:@"HNSreturn"] -> break; -> return;
		if(method == @selector(CSIF:))
			return [NSString stringWithFormat:@"if(%@) {\n%@}\n", [self makeObjectiveC:left], [self makeObjectiveC:[right objectAtIndex:0]]];
		if(method == @selector(CSIF:ELSE:))
			return [NSString stringWithFormat:@"if(%@) {\n%@}\nelse {\n%@}\n", [self makeObjectiveC:left], [self makeObjectiveC:[right objectAtIndex:0]], [self makeObjectiveC:[right objectAtIndex:1]]];
		if(method == @selector(CSSELECT:) || method == @selector(CSSELECT:DEFAULT:))
			{
			s=@"select()\n{\n";
			// add case labels incl. break; before next one
			if(method == @selector(CSSELECT:DEFAULT:))
				; // add default:
	        // add }
			return s;
			}
		e=[[NSStringFromSelector(method) componentsSeparatedByString:@":"] objectEnumerator];
		s=[NSString stringWithFormat:@"[%@ %@", [self makeObjectiveC:left], [e nextObject]];
		for(i=0; i<[right count]; i++)
			{ // add arguments
			if(i == 0)
				s=[s stringByAppendingFormat:@":%@", [self makeObjectiveC:[right objectAtIndex:i]]];	// first argument
			else
				{
				m=[e nextObject];
				if([m length] != 0)
					s=[s stringByAppendingFormat:@" %@:%@", m, [self makeObjectiveC:[right objectAtIndex:i]]];
				else
					s=[s stringByAppendingFormat:@", %@", [self makeObjectiveC:[right objectAtIndex:i]]];	// vararg
				}
			}
		return [s stringByAppendingString:@"]"];
		}
	if(left)
		return [self makeObjectiveC:left];	// left but no method
	if(method == @selector(getArgv))
		return @"argv;";	// special
	if(method == @selector(getSelf))
		return @"self";		// special
	if(method == @selector(getScript))
		return @"script";	// special
	if(method == @selector(setValueForName:to:))
		return [NSString stringWithFormat:@"%@=%@;\n", [right objectAtIndex:0], [self makeObjectiveC:[right objectAtIndex:1]]];
	if(method == @selector(valueForName:))
		return [NSString stringWithFormat:@"%@", [self makeObjectiveC:[right objectAtIndex:0]]];
	if(method == @selector(makeArray:))
		{ // build fresh array from contents
		s=[NSString stringWithFormat:@"[NSArray arrayWithObjects:"];
		for(i=0; i<[right count]; i++)
			{ // add arguments
			if(i == 0)
				s=[s stringByAppendingFormat:@"%@", [self makeObjectiveC:[right objectAtIndex:i]]];	// first argument
			else
				s=[s stringByAppendingFormat:@", %@", [self makeObjectiveC:[right objectAtIndex:i]]];
			}
		return [s stringByAppendingString:@", nil]"];
		}
	if(method == @selector(makeDictionary:))
		{ // build fresh dictionary from contents
		s=[NSString stringWithFormat:@"[NSDictionary dictionaryWithObjectsAndKeys:"];
		for(i=0; i<[right count]; i+=2)
			{ // add arguments
			if(i == 0)
				s=[s stringByAppendingFormat:@"%@, %@", [self makeObjectiveC:[right objectAtIndex:i+1]], [self makeObjectiveC:[right objectAtIndex:i]]];	// first argument
			else
				s=[s stringByAppendingFormat:@", %@, %@", [self makeObjectiveC:[right objectAtIndex:i+1]], [self makeObjectiveC:[right objectAtIndex:i]]];
			}
		return [s stringByAppendingString:@", nil]"];
		}
	for(i=0; i<[right count]; i++)
		{ // add scripts as separate statements
		NSString *r=[self makeObjectiveC:[right objectAtIndex:i]];
		if([r hasSuffix:@"}\n"])
			s=[s stringByAppendingFormat:@"%@", r];
		else
			s=[s stringByAppendingFormat:@"%@;\n", r];
		}
	return s;
}

@end

@implementation NSObject (HNSCocoaScript)

- (BOOL) boolValue
{
	[NSException raise:@"Value Error" format:@"Object has no Boolean value"];
	return NO;
}

- (int) intValue
{
	[NSException raise:@"Value Error" format:@"Object has no Integer value"];
	return 0;
}

- (long) longValue
{
	[NSException raise:@"Value Error" format:@"Object has no Integer value"];
	return 0;
}

- (double) doubleValue
{
	[NSException raise:@"Value Error" format:@"Object has no Double value"];
	return 0.0;
}

- (NSString *) stringValue
{
	[NSException raise:@"Value Error" format:@"Object has no String value"];
	return @"";
}

// this allows ANY object being passed to MsgBox function and the description will be shown

- (void) msgbox
{ // int NSRunAlertPanel(NSString *title, NSString *msg, NSString *defaultButton, NSString *alternateButton, NSString *otherButton, ...)
	NSRunAlertPanel(@"Message", @"%@", @"OK", nil, nil, [self description]);
}

- (id) CSIF:(HNSCocoaScript *) t
{ // conditional evaluation
	return [self boolValue]?[t eval]:nil;
}

- (id) CSIF:(HNSCocoaScript *) t ELSE:(HNSCocoaScript *) f
{ // conditional evaluation with alternative
	return [self boolValue]?[t eval]:[f eval];
}

- (id) CSSELECT:(NSDictionary *) t;
{ // selected evaluation
	id s=[t objectForKey:self];
	if(s)
		return [s eval];	// selected subscript
	return nil;		// no value
}

- (id) CSSELECT:(NSDictionary *) t DEFAULT:(HNSCocoaScript *) f;
{ // selected evaluation with default
	id s=[t objectForKey:self];
	if(s)
		return [s eval];	// selected subscript
	return [f eval];		// false part
}

#if 0
- (void)doesNotRecognizeSelector:(SEL)aSelector;
{
}

- (NSMethodSignature *)methodSignatureForSelector:(SEL)aSelector
{
	NSLog(@"methodSignatureForSelector %@", NSStringFromSelector(aSelector));
	return nil;
}

- (void)forwardInvocation:(NSInvocation *)anInvocation
{
	NSLog(@"forwardInvocation %@", anInvocation);
}
#endif

@end

@implementation NSNumber (HNSCocoaScript)

- (id) CSadd:(id) b
{ // b must be numeric as well
	return [NSNumber numberWithDouble:[self doubleValue]+[b doubleValue]];
}

- (id) CSsub:(id) b
{ // b must be numeric as well
	return [NSNumber numberWithDouble:[self doubleValue]-[b doubleValue]];
}

- (id) CSmul:(id) b
{ // b must be numeric as well
	return [NSNumber numberWithDouble:[self doubleValue]*[b doubleValue]];
}

- (id) CSdiv:(id) b
{ // b must be numeric as well
	return [NSNumber numberWithDouble:[self doubleValue]/[b doubleValue]];
}

- (id) CSand:(id) b
{ // b must be boolean as well
	return [NSNumber numberWithBool:([self boolValue] && [b boolValue])];
}

- (id) CSor:(id) b
{ // b must be boolean as well
	return [NSNumber numberWithBool:([self boolValue] || [b boolValue])];
}

- (id) CSxor:(id) b
{ // b must be boolean as well
	return [NSNumber numberWithBool:([self boolValue] != [b boolValue])];
}

- (id) CSchs
{ // change sign
	return [NSNumber numberWithDouble:-[self doubleValue]];
}

- (id) CSnot
{ // boolean not
	return [NSNumber numberWithBool:![self boolValue]];
}

- (id) CSle:(id) b;		// b must be number as well
{
	return [NSNumber numberWithBool:([self doubleValue] <= [b doubleValue])];
}

- (id) CSlt:(id) b;		// b must be number as well
{
	return [NSNumber numberWithBool:([self doubleValue] < [b doubleValue])];
}

- (id) CSge:(id) b;		// b must be number as well
{
	return [NSNumber numberWithBool:([self doubleValue] >= [b doubleValue])];
}

- (id) CSgt:(id) b;		// b must be number as well
{
	return [NSNumber numberWithBool:([self doubleValue] > [b doubleValue])];
}

- (id) CSeq:(id) b;		// b must be number as well
{
	return [NSNumber numberWithBool:([self doubleValue] == [b doubleValue])];
}

- (id) CSne:(id) b;		// b must be number as well
{
	return [NSNumber numberWithBool:([self doubleValue] != [b doubleValue])];
}

- (NSString *) str;
{
	return [NSString stringWithFormat:@"%lg", [self doubleValue]];
}

- (NSString *) makeObjectiveC;
{
	return [NSString stringWithFormat:@"[NSNumber numberWithDouble:%lg]", [self doubleValue]];
}

@end

@implementation NSString (HNSCocoaScript)

- (id) CSadd:(id) b
{ // b must be string as well
	return [self stringByAppendingString:b];
}

- (id) CSeq:(id) b;		// b must be string as well
{
	return [NSNumber numberWithBool:[self isEqualToString:b]];
}

- (id) CSne:(id) b;		// b must be string as well
{
	return [NSNumber numberWithBool:![self isEqualToString:b]];
}

- (NSString *) encodeAsHTML;
{ // encode all special characters as HTML
	// specials are " & < > and all with code >=0x80 (i.e. &amp; &#64235)
	return [[[NSString alloc] initWithData:[self dataUsingExtendedEncoding:NSHTMLStringEncoding] encoding:NSASCIIStringEncoding] autorelease];
}

- (NSString *) encodeAsURL;
{ // handle URL escapes
	NSMutableString *url=[[self mutableCopy] autorelease];
	[url replaceOccurrencesOfString:@"%" withString:@"%25" options:0 range:NSMakeRange(0, [url length])];
	[url replaceOccurrencesOfString:@" " withString:@"%20" options:0 range:NSMakeRange(0, [url length])];
	[url replaceOccurrencesOfString:@":" withString:@"%3a" options:0 range:NSMakeRange(0, [url length])];
	[url replaceOccurrencesOfString:@"/" withString:@"%2f" options:0 range:NSMakeRange(0, [url length])];
	[url replaceOccurrencesOfString:@";" withString:@"%3b" options:0 range:NSMakeRange(0, [url length])];
	[url replaceOccurrencesOfString:@"@" withString:@"%40" options:0 range:NSMakeRange(0, [url length])];
	return url;
}

- (void) showurl
{ // interpret string as URL and show
	[[NSWorkspace sharedWorkspace] openURL:[NSURL URLWithString:self]];
}

- (NSString *) makeObjectiveC;
{
	// handle escapes!
	return [NSString stringWithFormat:@"@\"%@\"", self];
}

- (id) len
{ // len("string") or "string".len
	return [NSNumber numberWithInt:[self length]];
}

- (id) left:(id) count
{ // "string".left(count) or left("string", count)
	int c=[count intValue];
	if(c < 0)
		return @"";
	if(c >= [self length])
		return self;	// limit
	return [self substringToIndex:c];
}

- (id) right:(id) count;
{ // "string".right(count)
	int c=[count intValue];
	if(c < 0)
		return @"";
	if(c >= [self length])
		return self;	// limit
	return [self substringFromIndex:[self length]-c];
}

- (id) mid:(id) pos
{ // "string".mid(pos)
	int p=[pos intValue];
	if(p < 0)
		return self;
	if(p > [self length])
		return @"";	// limit
	return [self substringFromIndex:p];
}

- (id) mid:(id) pos length:(id) count
{ // "string".mid(pos, length:count)
	int p=[pos intValue], c=[count intValue];
	if(p >= [self length] || c < 0)
		return @"";	// empty result
	if(p < 0)
		p=0;
	if(c > [self length]-p)
		c=[self length]-p;	// limit to what is available
	return [self substringWithRange:NSMakeRange(p, c)];
}

- (Class) findClass;
{ // "string".findClass
	return NSClassFromString(self);
}

@end

@implementation NSArray (HNSCocoaBasic)
- (id) CSadd:(id) b;	// b must be array as well
{
	return [self arrayByAddingObjectsFromArray:b];
}

- (unsigned) origin;	// array index origin - returns 0
{
	return 0;
}

- (NSString *) makeObjectiveC;
{
	// handle escapes!
	return [NSString stringWithFormat:@"[NSArray arrayWithObjects:first, second, ..., nil]"];
}

@end

@implementation NSDictionary (HNSCocoaBasic)
- (id) CSadd:(id) b;	// b must be dictionary as well
{
	NSEnumerator *enumerator=[b keyEnumerator];
	id key;
	NSMutableDictionary *d=[NSMutableDictionary dictionaryWithDictionary:self];	// make a copy
	while((key=[enumerator nextObject]))
		[d setObject:[b objectForKey:key] forKey:key];	// insert/replace
	return d;
}
@end

@implementation NSURL (HNSCocoaBasic)

- (void) showurl
{ // open URL
	[[NSWorkspace sharedWorkspace] openURL:self];
}

@end

@implementation NSInvocation (HNSCocoaBasic)

- (void)setVariableArgument:(void *) argumentLocation ofType:(enum _NSObjCValueType) argumentType;
{ // mangle argumentFrame
	NSLog(@"setVariableArgument: %@", self);
	NSLog(@"argumentLocation: %@", argumentLocation);
	NSLog(@"argumentType: %c", argumentType);
	NSLog(@"argumentFrame: %x", self->argumentFrame);
	NSLog(@"signature: %@", self->signature);
	NSLog(@"container: %@", self->container);
}

@end
